function [eigenvalues_,result,info] = check(M_, options_, oo_)
%[eigenvalues_,result,info] = check(M_, options_, oo_)
% Checks determinacy conditions by computing the generalized eigenvalues.
%
% INPUTS
% - M_            [structure]     MATLAB's structure describing the model
% - options_      [structure]     MATLAB's structure describing the current options
% - oo_           [structure]     MATLAB's structure containing the results
%
% OUTPUTS
% - eigenvalues_  [double]        vector, eigenvalues.
% - result        [integer]       logical scalar, equal to true if Blanchard and Kahn conditions are satisfied, false otherwise.
% - info          [integer]       scalar or vector, error code as returned by resol routine.

% Copyright © 2001-2025 Dynare Team
%
% This file is part of Dynare.
%
% Dynare is free software: you can redistribute it and/or modify
% it under the terms of the GNU General Public License as published by
% the Free Software Foundation, either version 3 of the License, or
% (at your option) any later version.
%
% Dynare is distributed in the hope that it will be useful,
% but WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
% GNU General Public License for more details.
%
% You should have received a copy of the GNU General Public License
% along with Dynare.  If not, see <https://www.gnu.org/licenses/>.


if ~options_.initval_file && M_.exo_nbr > 1
    oo_.exo_simul = ones(M_.maximum_lead+M_.maximum_lag+1,1)*oo_.exo_steady_state';
end

options_.order = 1;

if isempty(options_.qz_criterium)
    options_.qz_criterium = 1+1e-6;
end

oo_.dr=set_state_space(oo_.dr,M_);

[dr,info] = resol(1,M_,options_,oo_.dr ,oo_.steady_state, oo_.exo_steady_state, oo_.exo_det_steady_state);

if ~ismember(info(1),[0,2,3,4]) %exclude BK conditions and QZ failure (2), for 2 throw error later
    print_info(info, 0, options_);
end

eigenvalues_ = dr.eigval;
[m_lambda,i]=sort(abs(eigenvalues_));

result = isfield(dr,'edim') && M_.nsfwrd == dr.edim && dr.full_rank;

if ~options_.noprint
    skipline()
    disp('EIGENVALUES:')
    fprintf('%16s %16s %16s\n','Modulus','Real','Imaginary');
    z=[m_lambda real(eigenvalues_(i)) imag(eigenvalues_(i))]';
    fprintf('%16.4g %16.4g %16.4g\n',z);
    if isfield(dr,'edim')
        fprintf('\nThere are %d eigenvalue(s) larger than 1 in modulus ', dr.edim);
        fprintf('for %d forward-looking variable(s).', M_.nsfwrd);
        skipline()
        if result
            disp('The order and rank conditions are verified.')
        elseif M_.nsfwrd ~= dr.edim
            disp('The order condition is NOT verified.')
        else
            disp('The rank condition is NOT verified.')
        end
    end
    skipline()
end

if info(1)==2
    print_info(info, 0, options_); %print QZ error now
end
