/*
 * Copyright © 2003-2022 Dynare Team
 *
 * This file is part of Dynare.
 *
 * Dynare is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Dynare is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Dynare.  If not, see <https://www.gnu.org/licenses/>.
 */

#include <cassert>
#include <cmath>
#include <cstdlib>

#include "NumericalConstants.hh"

int
NumericalConstants::AddNonNegativeConstant(const string& iConst)
{
  if (auto iter = numConstantsIndex.find(iConst); iter != numConstantsIndex.end())
    return iter->second;

  auto id = static_cast<int>(mNumericalConstants.size());
  mNumericalConstants.push_back(iConst);
  numConstantsIndex[iConst] = id;

  /* Note that we allow underflows (will be converted to 0) and overflows (will
     be converted to Inf), as MATLAB and Octave do. As a consequence, we
     cannot use std::stod() here, since it does not allow distinguishing
     between underflow and overflow. */
  double val = strtod(iConst.c_str(), nullptr);

  assert(val >= 0 || isnan(val)); // Check we have a positive constant or a NaN
  double_vals.push_back(val);

  return id;
}

string
NumericalConstants::get(int ID) const
{
  assert(ID >= 0 && ID < static_cast<int>(mNumericalConstants.size()));
  return mNumericalConstants[ID];
}

double
NumericalConstants::getDouble(int ID) const
{
  assert(ID >= 0 && ID < static_cast<int>(double_vals.size()));
  return double_vals[ID];
}
