function plot_regimes(regimes,M_,options_)
% plot_regimes(regimes,M_,options_)
% Inputs: 
% - regimes     	[structure]     OccBin regime information
% - M_              [structure]     MATLAB's structure describing the model
% - options_        [structure]     MATLAB's structure containing the options

% Copyright © 2021-2023 Dynare Team
%
% This file is part of Dynare.
%
% Dynare is free software: you can redistribute it and/or modify
% it under the terms of the GNU General Public License as published by
% the Free Software Foundation, either version 3 of the License, or
% (at your option) any later version.
%
% Dynare is distributed in the hope that it will be useful,
% but WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
% GNU General Public License for more details.
%
% You should have received a copy of the GNU General Public License
% along with Dynare.  If not, see <https://www.gnu.org/licenses/>.

nperiods = size(regimes,2);
nconstr = length(fieldnames(regimes(1)))/2;
if nconstr ==1
    regime(1) = {'regime'};
    regimestart(1) = {'regimestart'};
else
    regime(1) = {'regime1'};
    regimestart(1) = {'regimestart1'};
    regime(2) = {'regime2'};
    regimestart(2) = {'regimestart2'};
end    

GraphDirectoryName = CheckPath('graphs',M_.dname);
fhandle = dyn_figure(options_.nodisplay,'Name',[M_.fname ': OccBin regimes']);

latexFolder = CheckPath('latex',M_.dname);
if options_.TeX && any(strcmp('eps',cellstr(options_.graph_format)))
    fidTeX = fopen([latexFolder '/' M_.fname '_occbin_regimes.tex'],'w');
    fprintf(fidTeX,'%% TeX eps-loader file generated by occbin.plot_regimes.m (Dynare).\n');
    fprintf(fidTeX,['%% ' datestr(now,0) '\n']);
    fprintf(fidTeX,' \n');
end

for k=1:nconstr
    subplot(nconstr,1,k)
    for t=1:nperiods
        nregimes_in_t = length(regimes(t).(regime{k}));
        start_periods = regimes(t).(regimestart{k});
        start_periods = [start_periods max(start_periods)];
        for r=1:nregimes_in_t
            isconstrained = regimes(t).(regime{k})(r);
            if isconstrained
                plot(t,start_periods(r),'*r')
                hold all,
                plot([t t],start_periods(r:r+1),'-r')
            else
                plot(t,start_periods(r),'ob')
                hold all,
                plot([t t],start_periods(r:r+1),'-b')
            end
        end
    end
    xlim([1 nperiods])
    title(['Regime ' int2str(k)])
    xlabel('Historic period')
    ylabel('Regime: expected start')
end
annotation('textbox',[.25,0,.15,.05],'String','Slack','Color','blue');
annotation('textbox',[.65,0,.2,.05],'String','Binding','Color','red');

dyn_saveas(fhandle,[GraphDirectoryName, filesep, M_.fname '_occbin_regimes'],options_.nodisplay,options_.graph_format);

if options_.TeX && any(strcmp('eps',cellstr(options_.graph_format)))
    fprintf(fidTeX,'\\begin{figure}[H]\n');
    fprintf(fidTeX,'\\centering \n');
    fprintf(fidTeX,'\\includegraphics[width=%2.2f\\textwidth]{%s_occbin_regimes}\n',options_.figures.textwidth,[GraphDirectoryName '/' M_.fname]);
    fprintf(fidTeX,'\\caption{OccBin: regime evolution over time.}\n');
    fprintf(fidTeX,'\\label{Fig:occbin_regimes}\n');
    fprintf(fidTeX,'\\end{figure}\n');
    fprintf(fidTeX,'\n');
    fclose(fidTeX);
end
