function [LIK,lik] = gaussian_filter(ReducedForm, Y, start, ParticleOptions, ThreadsOptions, options_, M_)

% Evaluates the likelihood of a non-linear model approximating the
% predictive (prior) and filtered (posterior) densities for state variables
% by gaussian distributions.
%
% INPUTS:
% - Reduced_Form     [struct]    MATLAB's structure describing the reduced form model.
% - Y                [double]    matrix of original observed variables.
% - start            [double]    structural parameters.
% - ParticleOptions  [struct]    MATLAB's structure describing options concerning particle filtering.
% - ThreadsOptions   [struct]    MATLAB's structure.
% - options_         [struct]    describing the options
% - M_               [struct]    describing the model
%
% OUTPUTS
% - LIK              [double]    scalar, likelihood
% - lik              [double]    vector, density of observations in each period.
%
% NOTES
% - The vector "lik" is used to evaluate the Jacobian of the likelihood.
% Gaussian approximation is done by:
%   - a spherical-radial cubature (ref: Arasaratnam & Haykin, 2009).
%   - a scaled unscented transform cubature (ref: Julier & Uhlmann 1995)
%   - Monte-Carlo draws from a multivariate gaussian distribution.
% First and second moments of prior and posterior state densities are computed
% from the resulting nodes/particles and allows to generate new distributions at the
% following observation.
%
% Pros: The use of nodes is much faster than Monte-Carlo Gaussian particle and standard particles
%       filters since it treats a lesser number of particles. Furthermore, in all cases, there is no need
%       of resampling.
% Cons: estimations may be biased if the model is truly non-Gaussian
%       since predictive and filtered densities are unimodal.

% Copyright © 2009-2026 Dynare Team
%
% This file is part of Dynare.
%
% Dynare is free software: you can redistribute it and/or modify
% it under the terms of the GNU General Public License as published by
% the Free Software Foundation, either version 3 of the License, or
% (at your option) any later version.
%
% Dynare is distributed in the hope that it will be useful,
% but WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
% GNU General Public License for more details.
%
% You should have received a copy of the GNU General Public License
% along with Dynare.  If not, see <https://www.gnu.org/licenses/>.

if ParticleOptions.pruning
    error('gaussian_filter: pruning is not yet implemented.')
end

% Set default
if isempty(start)
    start = 1;
end

sample_size = size(Y,2);
number_of_state_variables = length(ReducedForm.mf0);
number_of_observed_variables = length(ReducedForm.mf1);
number_of_particles = ParticleOptions.number_of_particles;

% compute gaussian quadrature nodes and weights on states and shocks
if ParticleOptions.distribution_approximation.cubature
    [nodes2, weights2] = spherical_radial_sigma_points(number_of_state_variables);
elseif ParticleOptions.distribution_approximation.unscented
    [nodes2, weights2] = unscented_sigma_points(number_of_state_variables,ParticleOptions.unscented);
else
    if ~ParticleOptions.distribution_approximation.montecarlo
        error('This approximation for the distribution is unknown!')
    end
end

if ParticleOptions.distribution_approximation.montecarlo
    set_dynare_seed_local_options([],false,'default');
end

% Get covariance matrices
H = ReducedForm.H;
if isempty(H)
    H = 0;
    H_lower_triangular_cholesky = 0;
else
    H_lower_triangular_cholesky = reduced_rank_cholesky(H)';
end

% Get initial condition for the state vector.
StateVectorMean = ReducedForm.StateVectorMean;
StateVectorVarianceSquareRoot = reduced_rank_cholesky(ReducedForm.StateVectorVariance)';
state_variance_rank = size(StateVectorVarianceSquareRoot,2);
Q_lower_triangular_cholesky = reduced_rank_cholesky(ReducedForm.Q)';

% Initialization of the likelihood.
lik  = NaN(sample_size,1);
LIK  = NaN;

for t=1:sample_size
    [PredictedStateMean, PredictedStateVarianceSquareRoot, StateVectorMean, StateVectorVarianceSquareRoot] = ...
        gaussian_filter_bank(ReducedForm, Y(:,t), StateVectorMean, StateVectorVarianceSquareRoot, Q_lower_triangular_cholesky, H_lower_triangular_cholesky, ...
                             H, ParticleOptions, options_, M_);
    if ParticleOptions.distribution_approximation.cubature || ParticleOptions.distribution_approximation.unscented
        StateParticles = bsxfun(@plus, StateVectorMean, StateVectorVarianceSquareRoot*nodes2');
        IncrementalWeights = gaussian_densities(Y(:,t), StateVectorMean, StateVectorVarianceSquareRoot, PredictedStateMean, ...
                                                PredictedStateVarianceSquareRoot, StateParticles, H, ReducedForm, ...
                                                options_, M_);
        SampleWeights = weights2.*IncrementalWeights;
    else
        StateParticles = bsxfun(@plus, StateVectorVarianceSquareRoot*randn(state_variance_rank, number_of_particles), StateVectorMean) ;
        IncrementalWeights = gaussian_densities(Y(:,t), StateVectorMean, StateVectorVarianceSquareRoot, PredictedStateMean, ...
                                                PredictedStateVarianceSquareRoot,StateParticles,H,ReducedForm, ...
                                                options_, M_);
        SampleWeights = IncrementalWeights/number_of_particles;
    end
    SampleWeights = SampleWeights + 1e-6*ones(size(SampleWeights, 1), 1);
    SumSampleWeights = sum(SampleWeights);
    lik(t) = log(SumSampleWeights);
    SampleWeights = SampleWeights./SumSampleWeights;
    if not(ParticleOptions.distribution_approximation.cubature || ParticleOptions.distribution_approximation.unscented)
        if (ParticleOptions.resampling.status.generic && neff(SampleWeights)<ParticleOptions.resampling.threshold*sample_size) || ParticleOptions.resampling.status.systematic
            StateParticles = resample(StateParticles', SampleWeights, ParticleOptions)';
            SampleWeights = ones(number_of_particles, 1)/number_of_particles;
        end
    end
    StateVectorMean = StateParticles*SampleWeights;
    temp = bsxfun(@minus, StateParticles, StateVectorMean);
    StateVectorVarianceSquareRoot = reduced_rank_cholesky(bsxfun(@times,SampleWeights',temp)*temp')';
end

LIK = -sum(lik(start:end));

function IncrementalWeights = gaussian_densities(obs,mut_t,sqr_Pss_t_t,st_t_1,sqr_Pss_t_t_1,particles,H,ReducedForm,options_, M_)
% IncrementalWeights = gaussian_densities(obs,mut_t,sqr_Pss_t_t,st_t_1,sqr_Pss_t_t_1,particles,H,ReducedForm,options_, M_)
% Elements to calculate the importance sampling ratio

% proposal density
proposal = probability2(mut_t, sqr_Pss_t_t, particles);

% prior density
prior = probability2(st_t_1, sqr_Pss_t_t_1, particles);

% likelihood
yt_t_1_i = measurement_equations(particles, ReducedForm, options_, M_);
likelihood = probability2(obs, sqrt(H), yt_t_1_i);

IncrementalWeights = likelihood.*prior./proposal;
