/*
 * This file shows how to use "system prior"-type prior restrictions as in
 * Michal Andrle/Miroslav Plašil (2018): "Econometrics with system priors",
 * Economics Letters, 172, pp. 134-137 during estimation based on
 * the baseline New Keynesian model of Jordi Galí (2015): Monetary Policy, Inflation,
 * and the Business Cycle, Princeton University Press, Second Edition, Chapter 3
 *
 * THIS MOD-FILE REQUIRES DYNARE 4.5 OR HIGHER
 *
 * Notes:
 *  - The estimation will automatically take the Gali_2015_prior_restrictions.m into
 *      account, which has the required name and format
 *  - Estimation is based on simulated data
 *  - The file also shows how to use a prior/posterior-function
 *
 * This implementation was written by Johannes Pfeifer. In case you spot mistakes,
 * email me at jpfeifer@gmx.de
 *
 * Please note that the following copyright notice only applies to this Dynare
 */

/*
 * Copyright © 2021-2023 Dynare Team
 *
 * This file is part of Dynare.
 *
 * Dynare is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Dynare is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Dynare.  If not, see <http://www.gnu.org/licenses/>.
 */

var pi          ${\pi}$                 (long_name='inflation')
    y_gap       ${\tilde y}$            (long_name='output gap')
    y_nat       ${y^{nat}}$             (long_name='natural output')      //(in contrast to the textbook defined in deviation from steady state)
    y           ${y}$                   (long_name='output')
    yhat        ${\hat y}$              (long_name='output deviation from steady state')
    r_nat       ${r^{nat}}$             (long_name='natural interest rate')
    i           ${i}$                   (long_name='nominal interrst rate')
    n           ${n}$                   (long_name='hours worked')
    nu          ${\nu}$                 (long_name='AR(1) monetary policy shock process')
    a           ${a}$                   (long_name='AR(1) technology shock process')
    z           ${z}$                   (long_name='AR(1) preference shock process')
    p           ${p}$                   (long_name='price level')
    w           ${w}$                   (long_name='nominal wage')
    c           ${c}$                   (long_name='consumption')
;

varexo  eps_a       ${\varepsilon_a}$       (long_name='technology shock')
        eps_nu  ${\varepsilon_\nu}$     (long_name='monetary policy shock')
        eps_z       ${\varepsilon_z}$   (long_name='preference shock innovation')
       ;

parameters alppha       ${\alpha}$     (long_name='capital share')
    betta               ${\beta}$       (long_name='discount factor')
    rho_a               ${\rho_a}$      (long_name='autocorrelation technology shock')
    rho_nu          ${\rho_{\nu}}$  (long_name='autocorrelation monetary policy shock')
    rho_z               ${\rho_{z}}$    (long_name='autocorrelation monetary demand shock')
    siggma              ${\sigma}$      (long_name='inverse EIS')
    varphi              ${\varphi}$     (long_name='inverse Frisch elasticity')
    phi_pi              ${\phi_{\pi}}$  (long_name='inflation feedback Taylor Rule')
    phi_y               ${\phi_{y}}$    (long_name='output feedback Taylor Rule')
    eta                 ${\eta}$        (long_name='semi-elasticity of money demand')
    epsilon             ${\epsilon}$    (long_name='demand elasticity')
    theta               ${\theta}$      (long_name='Calvo parameter')
    ;
%----------------------------------------------------------------
% Parametrization, p. 67  and p. 113-115
%----------------------------------------------------------------
siggma = 1;
varphi = 5;
phi_pi = 1.5;
phi_y  = 0.125;
theta = 3/4;
rho_nu =0.5;
rho_z  = 0.5;
rho_a  = 0.9;
betta  = 0.99;
eta  = 3.77; %footnote 11, p. 115
alppha = 1/4;
epsilon = 9;

%----------------------------------------------------------------
% First Order Conditions
%----------------------------------------------------------------

model(linear);
//Composite parameters
#Omega=(1-alppha)/(1-alppha+alppha*epsilon);        %defined on page 60
#psi_n_ya=(1+varphi)/(siggma*(1-alppha)+varphi+alppha);   %defined on page 62
#lambda=(1-theta)*(1-betta*theta)/theta*Omega;      %defined on page 61
#kappa=lambda*(siggma+(varphi+alppha)/(1-alppha));     %defined on page 63
[name='New Keynesian Phillips Curve eq. (22)']
pi=betta*pi(+1)+kappa*y_gap;
[name='Dynamic IS Curve eq. (23)']
y_gap=-1/siggma*(i-pi(+1)-r_nat)+y_gap(+1);
[name='Interest Rate Rule eq. (26)']
i=phi_pi*pi+phi_y*yhat+nu;
[name='Definition natural rate of interest eq. (24)']
r_nat=-siggma*psi_n_ya*(1-rho_a)*a+(1-rho_z)*z;
[name='Definition natural output, eq. (20)']
y_nat=psi_n_ya*a;
[name='Definition output gap']
y_gap=y-y_nat;
[name='Monetary policy shock']
nu=rho_nu*nu(-1)+eps_nu;
[name='TFP shock']
a=rho_a*a(-1)+eps_a;
[name='Production function (eq. 14)']
y=a+(1-alppha)*n;
[name='Preference shock, p. 54']
z=rho_z*z(-1) - eps_z;
[name='Output deviation from steady state']
yhat=y-steady_state(y);
[name='Definition price level']
pi=p-p(-1);
[name='resource constraint, eq. (12)']
y=c;
[name='FOC labor, eq. (2)']
w-p=siggma*c+varphi*n;
end;


shocks;
    var eps_nu = 0.0025^2; //1 standard deviation shock of 25 basis points, i.e. 1 percentage point annualized
end;

% simulate data
stoch_simul(periods=100,drop=0,irf=0) yhat;
% save data
datatomfile('sim_data',{'yhat'});

estimated_params;
theta,0.75,beta_pdf,0.5,0.1;
betta, beta_pdf, 0.993, 0.002;
alppha, beta_pdf, 0.25, 0.02;
end;

varobs yhat;

% Run prior function to get prior slope of the PC based on independent priors
hh_fig=figure('Name','Slope of the Phillips Curve');
prior_function(function='Gali_2015_PC_slope');
PC_slope_vec=cell2mat(oo_.prior_function_results(:,1));

optimal_bandwidth = mh_optimal_bandwidth(PC_slope_vec,length(PC_slope_vec),0,'gaussian');
[density(:,1),density(:,2)] = kernel_density_estimate(PC_slope_vec,512,length(PC_slope_vec),optimal_bandwidth,'gaussian');
figure(hh_fig)
subplot(3,1,1)
plot(density(:,1),density(:,2));
title('Prior')

% Run estimation with 1 observation to show effect of _prior_restriction .m
% on independent prior
estimation(datafile='sim_data',mode_compute=5,mh_replic=2001,mh_nblocks=1,diffuse_filter,nobs=1,mh_jscale=0.8);
posterior_function(function='Gali_2015_PC_slope');
PC_slope_vec=cell2mat(oo_.posterior_function_results(:,1));
optimal_bandwidth = mh_optimal_bandwidth(PC_slope_vec,length(PC_slope_vec),0,'gaussian');
[density(:,1),density(:,2)] = kernel_density_estimate(PC_slope_vec,512,length(PC_slope_vec),optimal_bandwidth,'gaussian');
figure(hh_fig)
subplot(3,1,2)
plot(density(:,1),density(:,2));
title('Updated Prior')


% Run estimation with full observations
estimation(datafile='sim_data',mode_compute=5,mh_replic=2001,mh_nblocks=1,diffuse_filter,nobs=100,mh_jscale=0.8);

posterior_function(function='Gali_2015_PC_slope');
PC_slope_vec=cell2mat(oo_.posterior_function_results(:,1));
optimal_bandwidth = mh_optimal_bandwidth(PC_slope_vec,length(PC_slope_vec),0,'gaussian');
[density(:,1),density(:,2)] = kernel_density_estimate(PC_slope_vec,512,length(PC_slope_vec),optimal_bandwidth,'gaussian');
figure(hh_fig)
subplot(3,1,3)
plot(density(:,1),density(:,2));
title('Posterior')
